bl_info = {
    "name": "Select similar dimensions",
    "author": "lemon",
    "version": (1, 0),
    "blender": (2, 80, 0),
    "location": "View3D > Select > Select similar dimensions",
    "description": "Select objects that have similar dimensions compared to the active object",
    "warning": "",
    "doc_url": "",
    "category": "Select Object",
}

import bpy

# loop over objects testing if their dimension is similar to a given object
# if similar, the objects are selected
def select_similar_dimensions(object, others, epsilon):
    
    # test the dimensions with an epsilon value
    def similar_dimensions(d, d_other, epsilon):
        delta = d - d_other
        return abs(delta.x) <= epsilon and abs(delta.y) <= epsilon and abs(delta.z) <= epsilon

    # get object dimension    
    dimensions = object.dimensions
    # loop over all objects
    for o in others:
        # select the one that are close in dimensions
        o.select_set(similar_dimensions(dimensions, o.dimensions, epsilon))


# the parts below are from "operator simple template"
# ...(adapted for our operator)

def main(context, epsilon):
    select_similar_dimensions(context.active_object, context.view_layer.objects, epsilon)


class SelectSimilarDimensions(bpy.types.Operator):
    """Select objects that have similar dimension compared to the active object"""
    bl_idname = "object.select_similar_dimensions"
    bl_label = "Select similar dimensions"
    bl_options = {'REGISTER', 'UNDO'}

    epsilon: bpy.props.FloatProperty(default=0.001)

    @classmethod
    def poll(cls, context):
        return context.active_object is not None

    def execute(self, context):
        main(context, self.epsilon)
        return {'FINISHED'}


def menu_func(self, context):
    self.layout.operator(SelectSimilarDimensions.bl_idname, text=SelectSimilarDimensions.bl_label)


def register():
    bpy.utils.register_class(SelectSimilarDimensions)
    bpy.types.VIEW3D_MT_select_object.append(menu_func)


def unregister():
    bpy.utils.unregister_class(SelectSimilarDimensions)
    bpy.types.VIEW3D_MT_select_object.remove(menu_func)


if __name__ == "__main__":
    register()
